/*
 * $Id$
 *
 * Copyright (c) 2005, 2006 Freie Universitaet Berlin.
 * All rights reserved.
 *
 * Written by Holger Weiss <holger@ZEDAT.FU-Berlin.DE>.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif			/* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if HAVE_STRINGS_H
#include <strings.h>
#endif			/* HAVE_STRINGS_H */

#include "conf.h"
#include "log.h"
#include "plugin.h"
#include "ups.h"
#include "text.h"
#include "zupsd.h"

RCSID("$Id$");

/*
 * This file parses zupsd.conf(5).
 *
 * See the manual page for the syntax of the configuration file.  Variables can
 * be of type STR (char *), INT (long) or BOO (signed char).  The generic code
 * checks the formal syntax, assigns the (specified or default) values and makes
 * sure the user specified INT variables as positive integers and BOO variables
 * as '0' or '1'.  Any further checks regarding variable values should be done
 * within the check_<module>_conf() functions.
 */

/*
 * HOWTO add a new variable:
 *
 * - add the variable to <module>_conf in conf.h
 * - add <MOD>_KEYWORD to the keywords enum
 * - add a row to the <module>_keywords table
 * - add a case to get_conf_ptr()
 */

/*
 * HOWTO add a new module:
 *
 * - provide two functions in <module>.c:
 *   1) int <module>_check(void *config);
 *   2) int <module>_test(void *config);
 * - export them via <module>.h, include that here
 * - add a struct <module>_conf typedef to conf.h
 * - add <MOD>_MODULE to the modules enum
 * - add a row to the modmap table
 * - add a <module>_keywords table
 * - add the following to get_conf_ptr():
 *   - the three <MOD>_{STR,INT,BOO}() macros
 *   - a <module>_conf pointer declaration named "<module>"
 *   - a case for each <module>_conf member
 * - provide two functions in conf.c:
 *   1) static bool new_<module>_conf(module *mod)
 *   this function
 *     - allocates memory for a "<module>_conf", pointed to by <module>
 *     - sets (at least) the following values:
 *       - mod->config = (void *) <module>;
 *       - mod->check = <module>_check;
 *       - mod->test = <module>_test;
 *       - mod->keywords = <module>_keywords;
 *       - mod->type = <MOD>_MODULE;
 *     - returns true on success and false on error
 *   2) static bool check_<module>_conf(const general_conf *general, void *conf)
 *   this function
 *     - optionally uses the INHERIT_GLOBAL_{STR,INT,BOO}() macros
 *     - optionally checks (some of) the <module>_conf values
 *     - returns true on success and false on error
 */

#define DFLT_PIDFILE    PIDFILE
#define DFLT_HOSTSFILE  DEFAULT_HOSTSFILE
#define DFLT_FACILITY   DEFAULT_FACILITY_NAME

#define UNSET_STR NULL
#define UNSET_INT -1
#define UNSET_BOO -1

#define INHERIT_GLOBAL_STR(local, global)                                      \
	do {                                                                   \
		if (local == UNSET_STR && global != UNSET_STR)                 \
			if ((local = strdup(global)) == NULL) {                \
				error("cannot allocate memory: %m");           \
				return false;                                  \
			}                                                      \
	} while (/* CONSTCOND */ 0)
#define INHERIT_GLOBAL_INT(local, global)              \
	if (local == UNSET_INT && global != UNSET_INT) \
		local = global
#define INHERIT_GLOBAL_BOO(local, global)              \
	if (local == UNSET_BOO && global != UNSET_BOO) \
		local = global

enum types {
	END,
	STR,
	INT,
	BOO
};

enum modules {
	GEN_MODULE = 1,
	UPS_MODULE,
	PLG_MODULE
};

enum keywords {
	GEN_CHECK_INTERVAL = 1,
	GEN_DEBUG_LEVEL,
	GEN_MAX_SHUTDOWNS,
	GEN_NUM_ALERT,
	GEN_NUM_CRITICAL,
	GEN_HOSTSFILE,
	GEN_PIDFILE,
	GEN_SHUTDOWN_CMD,
	GEN_SUICIDE_CMD,
	GEN_SYSLOG_FACILITY,
	GEN_COMMIT_SUICIDE,
	GEN_ALERT_THRESHOLD,
	GEN_EMAIL_INFOLEVEL,
	GEN_PAGER_INFOLEVEL,
	GEN_RETRIES,
	GEN_TIMEOUT,
	GEN_HOSTNAME,
	GEN_COMMUNITY,
	GEN_CONTACT_EMAIL,
	GEN_CONTACT_PAGER,
	GEN_EMAIL_CMD,
	GEN_PAGER_CMD,
	GEN_CHECK_CMD,
	GEN_TEST_CMD,
	UPS_ALERT_THRESHOLD,
	UPS_EMAIL_INFOLEVEL,
	UPS_PAGER_INFOLEVEL,
	UPS_RETRIES,
	UPS_TIMEOUT,
	UPS_HOSTNAME,
	UPS_COMMUNITY,
	UPS_EMAIL_CONTACT,
	UPS_PAGER_CONTACT,
	UPS_EMAIL_CMD,
	UPS_PAGER_CMD,
	PLG_HOSTNAME,
	PLG_CHECK_CMD,
	PLG_TEST_CMD
};

typedef struct {
	module *general;	/* first module parsed */
	module *current;	/* current/last module */
} config;

typedef struct {
	const char *name;
	const unsigned char type;
	bool (*new_conf)(module *mod);
	bool (*check_conf)(const general_conf *general, void *conf);
} modulemap;

static unsigned int num_modules = 0;

static bool new_general_conf(module *mod);
static bool new_ups_conf(module *mod);
static bool new_plugin_conf(module *mod);
static bool check_general_conf(const general_conf *unused UNUSED, void *conf);
static bool check_ups_conf(const general_conf *general, void *conf);
static bool check_plugin_conf(const general_conf *general, void *conf);

static const modulemap modmap[] = {
	{ "general", GEN_MODULE, new_general_conf, check_general_conf,  },
	{ "ups",     UPS_MODULE, new_ups_conf,     check_ups_conf,      },
	{ "plugin",  PLG_MODULE, new_plugin_conf,  check_plugin_conf,   },
	{  NULL,     END,        NULL,             NULL,                },
};

static const keyword general_keywords[] = {
	{ "hostsfile",       GEN_HOSTSFILE,       STR, DFLT_HOSTSFILE,  },
	{ "pidfile",         GEN_PIDFILE,         STR, DFLT_PIDFILE,    },
	{ "shutdown_cmd",    GEN_SHUTDOWN_CMD,    STR, NULL,            },
	{ "suicide_cmd",     GEN_SUICIDE_CMD,     STR, NULL,            },
	{ "syslog_facility", GEN_SYSLOG_FACILITY, STR, DFLT_FACILITY,   },
	{ "check_interval",  GEN_CHECK_INTERVAL,  INT, "30",            },
	{ "debug_level",     GEN_DEBUG_LEVEL,     INT, "0",             },
	{ "max_shutdowns",   GEN_MAX_SHUTDOWNS,   INT, "0",             },
	{ "num_alert",       GEN_NUM_ALERT,       INT, "0",             },
	{ "num_critical",    GEN_NUM_CRITICAL,    INT, "0",             },
	{ "commit_suicide",  GEN_COMMIT_SUICIDE,  BOO, "0",             },
	{ "hostname",        GEN_HOSTNAME,        STR, NULL,            },
	{ "community",       GEN_COMMUNITY,       STR, NULL,            },
	{ "email_contact",   GEN_CONTACT_EMAIL,   STR, NULL,            },
	{ "pager_contact",   GEN_CONTACT_PAGER,   STR, NULL,            },
	{ "email_cmd",       GEN_EMAIL_CMD,       STR, NULL,            },
	{ "pager_cmd",       GEN_PAGER_CMD,       STR, NULL,            },
	{ "alert_threshold", GEN_ALERT_THRESHOLD, INT, "0",             },
	{ "email_infolevel", GEN_EMAIL_INFOLEVEL, INT, "4",             },
	{ "pager_infolevel", GEN_PAGER_INFOLEVEL, INT, "2",             },
	{ "retries",         GEN_RETRIES,         INT, "5",             },
	{ "timeout",         GEN_TIMEOUT,         INT, "1000",          },
	{ "check_cmd",       GEN_CHECK_CMD,       STR, NULL,            },
	{ "test_cmd",        GEN_TEST_CMD,        STR, NULL,            },
	{  NULL,             END,                 END, NULL,            },
};

static const keyword ups_keywords[] = {
	{ "hostname",        UPS_HOSTNAME,        STR, NULL,            },
	{ "community",       UPS_COMMUNITY,       STR, "public",        },
	{ "email_contact",   UPS_EMAIL_CONTACT,   STR, NULL,            },
	{ "pager_contact",   UPS_PAGER_CONTACT,   STR, NULL,            },
	{ "email_cmd",       UPS_EMAIL_CMD,       STR, NULL,            },
	{ "pager_cmd",       UPS_PAGER_CMD,       STR, NULL,            },
	{ "alert_threshold", UPS_ALERT_THRESHOLD, INT, "0",             },
	{ "email_infolevel", UPS_EMAIL_INFOLEVEL, INT, "4",             },
	{ "pager_infolevel", UPS_PAGER_INFOLEVEL, INT, "2",             },
	{ "retries",         UPS_RETRIES,         INT, "5",             },
	{ "timeout",         UPS_TIMEOUT,         INT, "1000", /* 1s */ },
	{  NULL,             END,                 END, NULL,            },
};

static const keyword plugin_keywords[] = {
	{ "hostname",        PLG_HOSTNAME,        STR, NULL,            },
	{ "check_cmd",       PLG_CHECK_CMD,       STR, NULL,            },
	{ "test_cmd",        PLG_TEST_CMD,        STR, NULL,            },
	{  NULL,             END,                 END, NULL,            },
};

typedef union {
	conf_boo_t *bp;
	conf_int_t *ip;
	conf_str_t *cp;
} conf_ptr;

static conf_ptr get_conf_ptr(const module *mod, unsigned char key);
static bool new_module(config **conf, const char *buf);
static bool save_value(const module *mod, const keyword *kw, const char *buf);
static bool init_conf_values(const module *mod);
static bool check_config(const module *mod);
static bool set_defaults(const module *modules);
static void free_strings(const module *mod);
static inline size_t keyword_length(const char *buf);

static bool
new_general_conf(module *mod)
{
	general_conf *general;

	if ((general = malloc(sizeof(general_conf))) == NULL) {
		error("cannot allocate memory: %m");
		return false;
	}
	mod->config = (void *) general;
	mod->keywords = general_keywords;
	mod->type = GEN_MODULE;
	return true;
}

static bool
new_ups_conf(module *mod)
{
	ups_conf *ups;

	if ((ups = malloc(sizeof(ups_conf))) == NULL) {
		error("cannot allocate memory: %m");
		return false;
	}
	ups->notified_by_email = 0;
	ups->notified_by_pager = 0;
	mod->config = (void *) ups;
	mod->check = ups_check;
	mod->test = ups_test;
	mod->keywords = ups_keywords;
	mod->type = UPS_MODULE;
	return true;
}

static bool
new_plugin_conf(module *mod)
{
	plugin_conf *plugin;

	if ((plugin = malloc(sizeof(plugin_conf))) == NULL) {
		error("cannot allocate memory: %m");
		return false;
	}
	mod->config = (void *) plugin;
	mod->check = plugin_check;
	mod->test = plugin_test;
	mod->keywords = plugin_keywords;
	mod->type = PLG_MODULE;
	return true;
}

static bool
check_general_conf(const general_conf *unused UNUSED, void *conf)
{
#define USE_DEFAULT_IF_ZERO(v, k)                                              \
	do {                                                                   \
		if (v == 0)                                                    \
			for (i = 0; general_keywords[i].key != END; i++)       \
				if (general_keywords[i].key == k) {            \
					v = (conf_int_t)                       \
					    atoi(general_keywords[i].dflt);    \
					break;                                 \
				}                                              \
	} while (/* CONSTCOND */ 0)

	general_conf *general = (general_conf *) conf;
	int i;

	if (general->shutdown_cmd == NULL) {
		error("no shutdown command specified");
		return false;
	}
	if (general->syslog_facility != UNSET_STR &&
	    strcasecmp(general->syslog_facility, DEFAULT_FACILITY_NAME) &&
	    str2facility(general->syslog_facility) ==
	    str2facility(DEFAULT_FACILITY_NAME)) {
		error("unsupported syslog_facility '%s' specified",
		    general->syslog_facility);
		return false;
	}
	if (general->num_critical == UNSET_INT || general->num_critical == 0)
		general->num_critical = num_modules;
	if (general->num_alert == UNSET_INT || general->num_alert == 0)
		general->num_alert = num_modules;
	USE_DEFAULT_IF_ZERO(general->check_interval, GEN_CHECK_INTERVAL);

	debug("checked [general] section");
	return true;
}

static bool
check_ups_conf(const general_conf *general, void *conf)
{
	ups_conf *ups = (ups_conf *) conf;

	INHERIT_GLOBAL_STR(ups->hostname, general->hostname);
	INHERIT_GLOBAL_STR(ups->community, general->community);
	INHERIT_GLOBAL_STR(ups->email_contact, general->email_contact);
	INHERIT_GLOBAL_STR(ups->pager_contact, general->pager_contact);
	INHERIT_GLOBAL_STR(ups->email_cmd, general->email_cmd);
	INHERIT_GLOBAL_STR(ups->pager_cmd, general->pager_cmd);
	INHERIT_GLOBAL_INT(ups->alert_threshold, general->alert_threshold);
	INHERIT_GLOBAL_INT(ups->email_infolevel, general->email_infolevel);
	INHERIT_GLOBAL_INT(ups->pager_infolevel, general->pager_infolevel);
	INHERIT_GLOBAL_INT(ups->retries, general->retries);
	INHERIT_GLOBAL_INT(ups->timeout, general->timeout);

	if (ups->hostname == UNSET_STR) {
		error("no hostname specified for UPS module");
		return false;
	}
	if (ups->email_cmd != UNSET_STR && ups->email_contact == UNSET_STR &&
	    strstr(ups->email_cmd, "%r") != NULL ) {
		error("'%%r' used in email_cmd but email_contact not specified "
		    "for UPS module %s", ups->hostname);
		return false;
	}
	if (ups->pager_cmd != UNSET_STR && ups->pager_contact == UNSET_STR &&
	    strstr(ups->pager_cmd, "%r") != NULL) {
		error("'%%r' used in pager_cmd but pager_contact not specified "
		    "for UPS module %s", ups->hostname);
		return false;
	}
	if (ups->email_infolevel > 4) {
		error("email_infolevel must be between 0 and 4 for UPS module "
		    "%s", ups->hostname);
		return false;
	}
	if (ups->pager_infolevel > 4) {
		error("pager_infolevel must be between 0 and 4 for UPS module "
		    "%s", ups->hostname);
		return false;
	}
	if (ups->email_cmd == UNSET_STR) {
		if (ups->email_contact != UNSET_STR)
			warn("email_contact will be ignored since no email_cmd "
			    "is specified for %s", ups->hostname);
		if (ups->email_infolevel != UNSET_INT)
			warn("email_infolevel will be ignored since no "
			    "email_cmd is specified for %s", ups->hostname);
	}
	if (ups->pager_cmd == UNSET_STR) {
		if (ups->pager_contact != UNSET_STR)
			warn("pager_contact will be ignored since no pager_cmd "
			    "is specified for %s", ups->hostname);
		if (ups->pager_infolevel != UNSET_INT)
			warn("pager_infolevel will be ignored since no "
			    "pager_cmd is specified for %s", ups->hostname);
	}

	debug("checked [ups] module for %s", ups->hostname);
	return true;
}

static bool
check_plugin_conf(const general_conf *general, void *conf)
{
	plugin_conf *plugin = (plugin_conf *) conf;

	INHERIT_GLOBAL_STR(plugin->hostname, general->hostname);
	INHERIT_GLOBAL_STR(plugin->check_cmd, general->check_cmd);
	INHERIT_GLOBAL_STR(plugin->test_cmd, general->test_cmd);

	if (plugin->check_cmd == UNSET_STR) {
		error("no check_cmd specified for plugin module");
		return false;
	}
	if (plugin->test_cmd == UNSET_STR)
		if ((plugin->test_cmd = strdup(plugin->check_cmd)) == NULL) {
			error("cannot allocate memory: %m");
			return false;
		}
	if (plugin->hostname == UNSET_STR) {
		if (strstr(plugin->check_cmd, "%h") != NULL) {
			error("'%%h' used in check_cmd but hostname not "
			    "specified for plugin module");
			return false;
		}
		if (strstr(plugin->test_cmd, "%h") != NULL) {
			error("'%%h' used in test_cmd but hostname not "
			    "specified for plugin module");
			return false;
		}
	}

	debug("checked [plugin] module for %s", plugin->check_cmd);
	return true;
}

static conf_ptr
get_conf_ptr(const module *mod, unsigned char key)
{
#define MODULE_STR(module_conf_t, module, x)                                  \
	module = (module_conf_t *) mod->config;                               \
	p.cp = &module->x;                                                    \
	break
#define MODULE_INT(module_conf_t, module, x)                                  \
	module = (module_conf_t *) mod->config;                               \
	p.ip = &module->x;                                                    \
	break
#define MODULE_BOO(module_conf_t, module, x)                                  \
	module = (module_conf_t *) mod->config;                               \
	p.bp = &module->x;                                                    \
	break
/* general module macros */
#define GEN_STR(x) MODULE_STR(general_conf, general, x)
#define GEN_INT(x) MODULE_INT(general_conf, general, x)
#define GEN_BOO(x) MODULE_BOO(general_conf, general, x)
/* ups module macros */
#define UPS_STR(x) MODULE_STR(ups_conf, ups, x)
#define UPS_INT(x) MODULE_INT(ups_conf, ups, x)
#define UPS_BOO(x) MODULE_BOO(ups_conf, ups, x)
/* plugin module macros */
#define PLG_STR(x) MODULE_STR(plugin_conf, plugin, x)
#define PLG_INT(x) MODULE_INT(plugin_conf, plugin, x)
#define PLG_BOO(x) MODULE_BOO(plugin_conf, plugin, x)

	general_conf *general;
	ups_conf *ups;
	plugin_conf *plugin;
	conf_ptr p;

	switch (key) {

	/* general module keywords */
	case GEN_CHECK_INTERVAL:  GEN_INT(check_interval);
	case GEN_DEBUG_LEVEL:     GEN_INT(debug_level);
	case GEN_MAX_SHUTDOWNS:   GEN_INT(max_shutdowns);
	case GEN_NUM_ALERT:       GEN_INT(num_alert);
	case GEN_NUM_CRITICAL:    GEN_INT(num_critical);
	case GEN_HOSTSFILE:       GEN_STR(hostsfile);
	case GEN_PIDFILE:         GEN_STR(pidfile);
	case GEN_SHUTDOWN_CMD:    GEN_STR(shutdown_cmd);
	case GEN_SUICIDE_CMD:     GEN_STR(suicide_cmd);
	case GEN_SYSLOG_FACILITY: GEN_STR(syslog_facility);
	case GEN_COMMIT_SUICIDE:  GEN_BOO(commit_suicide);
	case GEN_ALERT_THRESHOLD: GEN_INT(alert_threshold);
	case GEN_EMAIL_INFOLEVEL: GEN_INT(email_infolevel);
	case GEN_PAGER_INFOLEVEL: GEN_INT(pager_infolevel);
	case GEN_RETRIES:         GEN_INT(retries);
	case GEN_TIMEOUT:         GEN_INT(timeout);
	case GEN_HOSTNAME:        GEN_STR(hostname);
	case GEN_COMMUNITY:       GEN_STR(community);
	case GEN_CONTACT_EMAIL:   GEN_STR(email_contact);
	case GEN_CONTACT_PAGER:   GEN_STR(pager_contact);
	case GEN_EMAIL_CMD:       GEN_STR(email_cmd);
	case GEN_PAGER_CMD:       GEN_STR(pager_cmd);
	case GEN_CHECK_CMD:       GEN_STR(check_cmd);
	case GEN_TEST_CMD:        GEN_STR(test_cmd);

	/* ups module keywords */
	case UPS_ALERT_THRESHOLD: UPS_INT(alert_threshold);
	case UPS_EMAIL_INFOLEVEL: UPS_INT(email_infolevel);
	case UPS_PAGER_INFOLEVEL: UPS_INT(pager_infolevel);
	case UPS_RETRIES:         UPS_INT(retries);
	case UPS_TIMEOUT:         UPS_INT(timeout);
	case UPS_HOSTNAME:        UPS_STR(hostname);
	case UPS_COMMUNITY:       UPS_STR(community);
	case UPS_EMAIL_CONTACT:   UPS_STR(email_contact);
	case UPS_PAGER_CONTACT:   UPS_STR(pager_contact);
	case UPS_EMAIL_CMD:       UPS_STR(email_cmd);
	case UPS_PAGER_CMD:       UPS_STR(pager_cmd);

	/* ups module keywords */
	case PLG_HOSTNAME:        PLG_STR(hostname);
	case PLG_CHECK_CMD:       PLG_STR(check_cmd);
	case PLG_TEST_CMD:        PLG_STR(test_cmd);

	default:
		/* should never happen */
		error("internal error: unknown keyword");
		p.bp = (conf_boo_t *) NULL;
	}

	return p;
}

module *
readconf(const char *file)
{
#define FREE(x)                                                                \
	do {                                                                   \
		if (x != NULL) {                                               \
			free_modules(x->general);                              \
			free(x);                                               \
		}                                                              \
	} while (/* CONTCOND*/ 0)

	config *conf = NULL;
	module *mod;
	FILE *fp;
	size_t n, len;
	unsigned int line, i;
	char buf[MAXLINELEN];
	char *p;

	debug("reading configuration file %s", file);

	if ((fp = fopen(file, "r")) == NULL) {
		error("cannot open %s: %m", file);
		return NULL;
	}
	for (n = 0, line = 1; fgets(buf + n, MAXLINELEN - n, fp) != NULL &&
	    (n = strlen(buf)); line++) {
		if (buf[n - 1] != '\n' && n == MAXLINELEN - 1) {
			error("line too long: %s:%d", file, line);
			FREE(conf);
			return NULL;
		}
		if (n >= 2 && buf[n - 2] == '\\') {
			/*
			 * The current line ends with a backslash, we'll append
			 * the following line to buf.  The backslash and newline
			 * will be overwritten due to 'n -= 2'.
			 */
			n -= 2;
			continue;
		}
		if (buf[n - 1] == '\n')
			buf[n - 1] = '\0';	/* delete newline */
		n = 0;
		/* skip whitespace at beginning of line */
		p = skip_whitespace(buf);
		if (*p == '\0' || *p == '#')
			continue;	/* skip empty line */
		if (*p == '[') {
			if (!new_module(&conf, p)) {
				error("module name parse error at %s:%d", file,
				    line);
				FREE(conf);
				return NULL;
			}
			continue;
		}
		if (conf == NULL) {
			error("%s must begin with '[general]' specification",
			   file);
			return NULL;
		}
		/* okay, we have conf->current and something to parse */
		len = keyword_length(p);
		for (i = 0; conf->current->keywords[i].type != END; i++) {
			if (strlen(conf->current->keywords[i].name) == len &&
			    !strncasecmp(conf->current->keywords[i].name, p,
			    len)) {
				/* skip whitespace following keyword */
				p = skip_whitespace(p + len);
				if (*p != '=') {
					error("no '=' operator following '%s' "
					    "keyword at %s:%d",
					    conf->current->keywords[i].name,
					    file, line);
					FREE(conf);
					return NULL;
				}
				/* skip whitespace following '=' */
				p = skip_whitespace(++p);
				if (!save_value(conf->current,
				    &conf->current->keywords[i], p)) {
					error("parse error for value of '%s' "
					    "keyword at %s:%d",
					    conf->current->keywords[i].name,
					    file, line);
					FREE(conf);
					return NULL;
				}
				break;
			}
		}
		if (conf->current->keywords[i].type == END) {
			p[len] = '\0';
			error("unknown keyword '%s' at %s:%d", p, file, line);
			FREE(conf);
			return NULL;
		}
	}
	if (ferror(fp) || !feof(fp)) {
		error("error while reading %s", file);
		FREE(conf);
		return NULL;
	}
	if (fclose(fp) != 0) {
		error("cannot close %s: %m", file);
		FREE(conf);
		return NULL;
	}

	if (!num_modules) {
		error("no modules specified");
		FREE(conf);
		return NULL;
	}
	if (!check_config(conf->general) || !set_defaults(conf->general)) {
		FREE(conf);
		return NULL;
	}
	mod = conf->general;
	free(conf);

	debug("done reading configuration file");
	return mod;
}

void
free_modules(module *modules)
{
#define FREE_MODULE(m) free_strings(m), free(m->config), free(m)
	module *mod, *next;

	if (modules == NULL)
		return;
	for (mod = modules->next; mod != NULL; mod = next) {
		next = mod->next;
		FREE_MODULE(mod);
	}
	mod = modules;	/* general module */
	FREE_MODULE(mod);

	debug("free(3)d configuration data");
}

static bool
new_module(config **conf, const char *buf)
{
	module *mod;
	size_t len;
	int i;

	if (*buf != '[') {
		/* should never happen */
		error("internal error: not a module specification");
		return false;
	}
	/* skip whitespace following '[' */
	buf = skip_whitespace(++buf);
	len = keyword_length(buf);
	for (i = 0; modmap[i].type != END; i++) {
		if (strlen(modmap[i].name) == len &&
		    !strncasecmp(modmap[i].name, buf, len))
			break;
	}
	if (modmap[i].type == END) {
		error("unknown module name");
		return false;
	}

	debug("adding [%s] module data structure", modmap[i].name);
	if (*conf == NULL) {
		/* TODO: get rid of the conf type, it's not needed */
		if (modmap[i].type != GEN_MODULE) {
			error("configuration must begin with '[general]' "
			    "section");
			return false;
		}
		if ((*conf = malloc(sizeof(config))) == NULL) {
			error("cannot allocate memory: %m");
			return false;
		}
		(*conf)->general = NULL;
	} else if (modmap[i].type == GEN_MODULE) {
		error("only one '[general]' section allowed");
		return false;
	}
	if ((mod = malloc(sizeof(module))) == NULL) {
		error("cannot allocate memory: %m");
		return false;
	}
	mod->next = NULL;
	if (!(modmap[i].new_conf)(mod))
		return false;
	if (!init_conf_values(mod))
		return false;

	if ((*conf)->general == NULL)
		(*conf)->general = mod;
	else {	/* (*conf)->current != NULL */
		(*conf)->current->next = mod;
		num_modules++;
	}
	(*conf)->current = mod;
	return true;
}

static bool
save_value(const module *mod, const keyword *kw, const char *buf)
{
	conf_ptr p;
	int i = 0;
	size_t len = strlen(buf);
	const char *in = buf;
	char *out, *end;
	char quote_char = 0;
	bool escaped = 0;

	if (*in == '"' || *in == '\'') {
		quote_char = *in;
		in++, i++;
	}
	if ((!quote_char && len < 1) || len < 3) {
		error("no value specified");
		return false;
	}
	p = get_conf_ptr(mod, kw->key);
	if (p.bp == (conf_boo_t *) NULL)
		return false;

	switch (kw->type) {
	case STR:
		if (*p.cp != UNSET_STR) {
			error("variable was specified already");
			return false;
		}
		if ((*p.cp = malloc(len + 1)) == NULL) {
			error("cannot allocate memory: %m");
			return false;
		}
		for (out = *p.cp; *in != '\0'; in++, out++, i++) {
			if (*in == '\\') {
				in++, i++;
				if (*in == '\0') {
					error("unexpected backslash");
					free(*p.cp);
					*p.cp = NULL;
					return false;
				}
				escaped = 1;
			}
			if (escaped)
				escaped = 0;
			else if (quote_char) {
				if (*in == quote_char) {
					i++;
					break;
				}
			} else if (*in == ' ' || *in == '#')
				break;
			*out = *in;
		}
		if (quote_char && *in != quote_char) {
			error("quotation not closed");
			free(*p.cp);
			*p.cp = NULL;
			return false;
		}
		*out = '\0';
		end = (char *) buf + i;
		debug("saved %s = \"%s\"", kw->name, *p.cp);
		break;
	case INT:
		if (*p.ip != UNSET_INT) {
			error("variable was specified already");
			return false;
		}
		if (*in == '-') {
			error("value must be a positive integer");
			return false;
		}
		*p.ip = (conf_int_t) strtol(in, &end, 10);
		if (in == end) {
			error("unexpected non-integer value");
			return false;
		}
		if (quote_char) {
			if (*end != quote_char) {
				error("quotation not closed");
				return false;
			}
			end++;
		}
		debug("saved %s = %ld", kw->name, *p.ip);
		break;
	case BOO:
		if (*p.bp != UNSET_BOO) {
			error("variable was specified already");
			return false;
		}
		if (*in == '1')
			*p.bp = true;
		else if (*in == '0')
			*p.bp = false;
		else {
			error("boolean value must be '0' or '1'");
			return false;
		}
		if (quote_char && (*(++in) != quote_char)) {
			error("quotation not closed");
			return false;
		}
		end = (char *) ++in;
		debug("saved %s = %d", kw->name, (int) *p.bp);
		break;
	default:
		/* should never happen */
		error("internal error: unknown keyword type");
		return false;
	}

	end = skip_whitespace(end);
	if (*end != '#' && *end != '\0') {
		error("unexpected stuff following value: '%s'", end);
		return false;
	}
	return true;
}

static bool
init_conf_values(const module *mod)
{
	conf_ptr p;
	int i;

	for (i = 0; mod->keywords[i].type != END; i++)
		switch(mod->keywords[i].type) {
		case STR:
			p = get_conf_ptr(mod, mod->keywords[i].key);
			if (p.bp == (conf_boo_t *) NULL)
				return false;
			*p.cp = UNSET_STR;
			break;
		case INT:
			p = get_conf_ptr(mod, mod->keywords[i].key);
			if (p.bp == (conf_boo_t *) NULL)
				return false;
			*p.ip = UNSET_INT;
			break;
		case BOO:
			p = get_conf_ptr(mod, mod->keywords[i].key);
			if (p.bp == (conf_boo_t *) NULL)
				return false;
			*p.bp = UNSET_BOO;
			break;
		default:
			/* should never happen */
			error("internal error: unknown keyword type");
			return false;
		}

	debug("initialized module configuration variables");
	return true;
}

static bool
check_config(const module *modules)
{
	const module *mod;
	int i;

	for (mod = modules; mod != NULL; mod = mod->next)
		for (i = 0; modmap[i].type != END; i++)
			if (mod->type == modmap[i].type) {
				if (!(modmap[i].check_conf)(modules->config,
				    mod->config))
					return false;
				break;
			}

	debug("done checking configuration");
	return true;
}

static bool
set_defaults(const module *modules)
{
	const module *mod;
	conf_ptr p;
	int i;

	for (mod = modules; mod != NULL; mod = mod->next)
		for (i = 0; mod->keywords[i].type != END; i++)
			switch(mod->keywords[i].type) {
			case STR:
				p = get_conf_ptr(mod, mod->keywords[i].key);
				if (p.bp == (conf_boo_t *) NULL)
					return false;
				if (*p.cp == UNSET_STR &&
				    mod->keywords[i].dflt != NULL && (*p.cp =
				    strdup(mod->keywords[i].dflt)) == NULL) {
					error("cannot allocate memory: %m");
					return false;
				}
				break;
			case INT:
				p = get_conf_ptr(mod, mod->keywords[i].key);
				if (p.bp == (conf_boo_t *) NULL)
					return false;
				if (*p.ip == UNSET_INT &&
				    mod->keywords[i].dflt != NULL)
					*p.ip = (conf_int_t)
					    atoi(mod->keywords[i].dflt);
				break;
			case BOO:
				p = get_conf_ptr(mod, mod->keywords[i].key);
				if (p.bp == (conf_boo_t *) NULL)
					return false;
				if (*p.bp == UNSET_BOO &&
				    mod->keywords[i].dflt != NULL)
					*p.bp = (conf_boo_t)
					    atoi(mod->keywords[i].dflt);
				break;
			default:
				/* should never happen */
				error("internal error: unknown keyword type");
				return false;
			}

	debug("default values set for unspecified keywords");
	return true;
}

static void
free_strings(const module *mod)
{
	conf_ptr p;
	int i;

	for (i = 0; mod->keywords[i].type != END; i++)
		if (mod->keywords[i].type == STR) {
			p = get_conf_ptr(mod, mod->keywords[i].key);
			if (*p.cp != NULL)
				free(*p.cp);
		}
}

static inline size_t
keyword_length(const char *buf)
{
	size_t length;

	for (length = 0; *buf != '=' && *buf != ']' && *buf != ' ' &&
	    *buf != '\t' && *buf != '\0'; buf++, length++)
		continue;
	return length;
}
