/*
 * $Id$
 *
 * Copyright (c) 2005, 2006, 2007 Freie Universitaet Berlin.
 * All rights reserved.
 *
 * Written by Holger Weiss <holger@ZEDAT.FU-Berlin.DE>.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif			/* HAVE_CONFIG_H */

#include <errno.h>
#include <stdio.h>
#include <string.h>
#if HAVE_STRINGS_H
#include <strings.h>
#endif			/* HAVE_STRINGS_H */
#include <syslog.h>

#include "log.h"
#include "zupsd.h"

RCSID("$Id$");

extern int errno;

static char progname[64];
static bool use_syslog = false;
static bool use_stderr = false;

static void vlog(const char *file, int line, int pri, const char *fmt, va_list args);
static inline const char *priority2str(int priority);

static void
vlog(const char *file, int line, int pri, const char *fmt, va_list args)
{
#define LOGPRELEN   64
#define LOGBUFLEN  512
#define TOOLONG   "[message too long]"
	size_t n;
	char prefix[LOGPRELEN], buf[LOGBUFLEN];
	const char *in;
	char *out;

	(void) strncpy(prefix, priority2str(pri), sizeof(prefix) - 1);
	prefix[sizeof(prefix) - 1] = '\0';

	for (in = fmt, out = buf, n = 0; *in != '\0' && n <= sizeof(buf) - 2;
	    in++, out = buf + n) {
		if (*in != '%') {
			*out = *in;
			n++;
			continue;
		}
		switch (*++in) {
		case 'i':
			n += snprintf(out, sizeof(buf) - n, "%i",
			    va_arg(args, int));
			break;
		case 'd':
			n += snprintf(out, sizeof(buf) - n, "%d",
			    va_arg(args, int));
			break;
		case 'o':
			n += snprintf(out, sizeof(buf) - n, "%o",
			    va_arg(args, int));
			break;
		case 'x':
			n += snprintf(out, sizeof(buf) - n, "%x",
			    va_arg(args, int));
			break;
		case 'X':
			n += snprintf(out, sizeof(buf) - n, "%X",
			    va_arg(args, int));
			break;
		case 'u':
			n += snprintf(out, sizeof(buf) - n, "%u",
			    va_arg(args, unsigned int));
			break;
		case 'e':
			n += snprintf(out, sizeof(buf) - n, "%e",
			    va_arg(args, double));
			break;
		case 'E':
			n += snprintf(out, sizeof(buf) - n, "%E",
			    va_arg(args, double));
			break;
		case 'f':
			n += snprintf(out, sizeof(buf) - n, "%f",
			    va_arg(args, double));
			break;
		case 'g':
			n += snprintf(out, sizeof(buf) - n, "%g",
			    va_arg(args, double));
			break;
		case 'G':
			n += snprintf(out, sizeof(buf) - n, "%G",
			    va_arg(args, double));
			break;
		case 'c':
			n += snprintf(out, sizeof(buf) - n, "%c",
			    va_arg(args, int));
			break;
		case 's':
			n += snprintf(out, sizeof(buf) - n, "%s",
			    va_arg(args, char *));
			break;
		case 'm':
			n += snprintf(out, sizeof(buf) - n, "%s",
			    strerror(errno));
			break;
		case 'l':
			switch (*++in) {
			case 'd':
				n += snprintf(out, sizeof(buf) - n, "%ld",
				    va_arg(args, long));
				break;
			case 'u':
				n += snprintf(out, sizeof(buf) - n, "%lu",
				    va_arg(args, unsigned long));
				break;
			default:
				*out++ = 'l';
				*out = *in;
				n += 2;
			}
			break;
		default:
			*out = *in;
			n++;
		}
	}

	if (n >= sizeof(buf)) {
		n = sizeof(buf) - 1 - strlen(TOOLONG);
		(void) strncpy(buf + n, TOOLONG, sizeof(buf) - 1 - n);
		buf[sizeof(buf) - 1] = '\0';
	} else
		buf[n] = '\0';

	if (use_syslog) {
		if (file != NULL && line)
			syslog(pri, "[%s] (%s:%d) %s", prefix, file, line, buf);
		else
			syslog(pri, "[%s] %s", prefix, buf);
	}
	if (use_stderr) {
		if (file != NULL && line)
			(void) fprintf(stderr, "%s: [%s] (%s:%d) %s\n",
			    progname, prefix, file, line, buf);
		else
			(void) fprintf(stderr, "%s: [%s] %s\n", progname,
			    prefix, buf);

		(void) fflush(stderr);
	}
}

void
open_log(const char *ident, int facility, int output)
{
	if (output & ZUPSD_SYSLOG) {
		openlog(ident, LOG_CONS | LOG_NDELAY | LOG_PID, facility);
		use_syslog = true;
	}
	if (output & ZUPSD_STDERR)
		use_stderr = true;

	(void) strncpy(progname, ident, sizeof(progname) - 1);
	progname[sizeof(progname) - 1] = '\0';
}

void
close_log(void)
{
	if (use_syslog) {
		closelog();
		use_syslog = false;
	}
	if (use_stderr)
		use_stderr = false;
}

void
reopen_log(const char *ident, int facility, int output)
{
	close_log();
	open_log(ident, facility, output);
}

int
str2facility(const char *facility)
{
#ifndef DEFAULT_FACILITY
#define DEFAULT_FACILITY LOG_DAEMON
#endif
	static const map facilities[] = {
		{ "DAEMON", LOG_DAEMON, },
		{ "LOCAL0", LOG_LOCAL0, },
		{ "LOCAL1", LOG_LOCAL1, },
		{ "LOCAL2", LOG_LOCAL2, },
		{ "LOCAL3", LOG_LOCAL3, },
		{ "LOCAL4", LOG_LOCAL4, },
		{ "LOCAL5", LOG_LOCAL5, },
		{ "LOCAL6", LOG_LOCAL6, },
		{ "LOCAL7", LOG_LOCAL7, },
		{ "USER",   LOG_USER,   },
		{  NULL,    0,          },
	};
	int i;

	for (i = 0; facilities[i].str != NULL; i++)
		if (!strcasecmp(facilities[i].str, facility))
			return facilities[i].num;
	return DEFAULT_FACILITY;
}

static inline const char *
priority2str(int priority)
{
	static const map priorities[] = {
		{ "EMERG",   LOG_EMERG,   },
		{ "ALERT",   LOG_ALERT,   },
		{ "CRIT",    LOG_CRIT,    },
		{ "ERR",     LOG_ERR,     },
		{ "WARNING", LOG_WARNING, },
		{ "NOTICE",  LOG_NOTICE,  },
		{ "INFO",    LOG_INFO,    },
		{ "DEBUG",   LOG_DEBUG,   },
		{  NULL,     0,           },
	};
	int i;

	for (i = 0; priorities[i].str != NULL; i++)
		if (priority == priorities[i].num)
			return priorities[i].str;
	return "unknown priority";
}

#if USE_LOG_MACROS
void
_log(const char *file, int line, int pri, const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vlog(file, line, pri, fmt, args);
	va_end(args);
}
#else
void
debug(const char *fmt, ...)
{
	va_list args;

	if (!debugging)
		return;
	va_start(args, fmt);
	vlog(NULL, 0, LOG_DEBUG, fmt, args);
	va_end(args);
}

void
info(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vlog(NULL, 0, LOG_INFO, fmt, args);
	va_end(args);
}

void
notice(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vlog(NULL, 0, LOG_NOTICE, fmt, args);
	va_end(args);
}

void
warn(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vlog(NULL, 0, LOG_WARNING, fmt, args);
	va_end(args);
}

void
error(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vlog(NULL, 0, LOG_ERR, fmt, args);
	va_end(args);
}

void
crit(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vlog(NULL, 0, LOG_CRIT, fmt, args);
	va_end(args);
}

void
alert(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vlog(NULL, 0, LOG_ALERT, fmt, args);
	va_end(args);
}
#endif			/* USE_LOG_MACROS */
