/*
 * $Id$
 *
 * Copyright (c) 2005, 2006, 2007 Freie Universitaet Berlin.
 * All rights reserved.
 *
 * Written by Holger Weiss <holger@ZEDAT.FU-Berlin.DE>.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif			/* HAVE_CONFIG_H */

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif			/* HAVE_SYS_TYPES_H */
#include <signal.h>
#include <stdlib.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#endif			/* HAVE_UNISTD_H */

#include "command.h"
#include "log.h"
#include "notify.h"
#include "sig.h"
#include "text.h"
#include "zupsd.h"

RCSID("$Id$");

static void setup_notification_signals(void);
RETSIGTYPE notification_sighandler(int sig);

pid_t
notify(const char *notify_cmd, const char *recipients, const char *message)
{
	pid_t pid;
	int status;
	char *tmp, *cmd;

	block_signals(true);
	if ((pid = fork()) == -1) {
		error("cannot fork notify handling process: %m");
		block_signals(false);
		return pid;
	} else if (pid) {
		block_signals(false);
		return pid;	/* parent returns */
	}
	setup_notification_signals();
	block_signals(false);

	if ((tmp = replace(notify_cmd, "%m", message)) == NULL)
		exit(EXIT_FAILURE);
	if (recipients != NULL) {
		if ((cmd = replace(tmp, "%r", recipients)) == NULL)
			exit(EXIT_FAILURE);
		free(tmp);
	} else
		cmd = tmp;
	if ((status = run(cmd)) != 0) {
		error("notification command '%s' failed with status %d", cmd,
		    status);
		exit(EXIT_FAILURE);
	}
	if (recipients)
		info("sent notification to %s", recipients);
	else
		debug("executed notification command '%s'", cmd);

	free(cmd);
	exit(EXIT_SUCCESS);
}

static void
setup_notification_signals(void)
{
	/* TODO: use sigaction(2) (if available) */
	signal(SIGCHLD, SIG_DFL);
	signal(SIGPIPE, SIG_IGN);	/* ignore SIGPIPE */
	signal(SIGTSTP, SIG_IGN);	/* ignore TTY signals */
	signal(SIGTTOU, SIG_IGN);
	signal(SIGTTIN, SIG_IGN);
	signal(SIGHUP,  SIG_IGN);
	signal(SIGUSR1, notification_sighandler);
	signal(SIGUSR2, notification_sighandler);
	signal(SIGTERM, notification_sighandler);
	signal(SIGINT,  notification_sighandler);
}

RETSIGTYPE
notification_sighandler(int sig)
{
	setup_notification_signals();

	/*
	 * For convenience, we'll forward handled signals to the shutdown
	 * parent, so that the user won't have to distinguish between the
	 * shutdown parent and the child processes.  Apart from that, we'll
	 * ignore SIGUSR1 and SIGUSR2.  For all other handled signals, we'll
	 * apply the default action to ourselves, which should be termination.
	 */
	(void) kill(getppid(), sig);
	if (sig != SIGUSR1 && sig != SIGUSR2) {
		info("received %s, terminating", signal2str(sig));
		signal(sig, SIG_DFL);
		(void) raise(sig);	/* default action */
	}
}
