/*
 * $Id$
 *
 * Copyright (c) 2005, 2006, 2007 Freie Universitaet Berlin.
 * All rights reserved.
 *
 * Written by Holger Weiss <holger@ZEDAT.FU-Berlin.DE>.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif			/* HAVE_CONFIG_H */

#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif			/* HAVE_SYS_TYPES_H */
#if HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif			/* HAVE_SYS_WAIT_H */
#include <errno.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#endif			/* HAVE_UNISTD_H */

#include "command.h"
#include "conf.h"	/* general_conf, MAXLINELEN */
#include "log.h"
#include "sig.h"
#include "shutdown.h"
#include "text.h"
#include "zupsd.h"

RCSID("$Id$");

extern int errno;

/* static volatile variables used in sighandler */
static volatile sig_atomic_t shutdown_immediately = 0;
static volatile sig_atomic_t shutdowns_running = 0;
static volatile sig_atomic_t shutdown_termination_running = 0;

static pid_t run_shutdown(const char *shutdown_cmd, const char *hostname,
                          const char *type);
static inline size_t token_length(const char *buf);
static void wait_for_shutdown_child(void);
static void setup_shutdown_signals(void);
static void setup_shutdown_child_signals(void);
RETSIGTYPE shutdown_sighandler(int sig);
RETSIGTYPE shutdown_child_sighandler(int sig);

host *
read_hosts(const char *file)
{
	host *list = NULL, *current, *prev, *tmp;
	FILE *fp;
	size_t n, len;
	unsigned int line;
	char buf[MAXLINELEN];
	char *p, *end;

	debug("reading hosts file %s", file);

	if ((fp = fopen(file, "r")) == NULL) {
		error("cannot open %s: %m", file);
		return NULL;
	}
	for (n = 0, line = 1; fgets(buf + n, MAXLINELEN - n, fp) != NULL &&
	    (n = strlen(buf)); line++) {
		if (buf[n - 1] != '\n' && n == MAXLINELEN - 1) {
			error("line too long: %s:%d", file, line);
			if (list != NULL)
				free_hosts(list);
			return NULL;
		}
		if (n >= 2 && buf[n - 2] == '\\') {
			/*
			 * The current line ends with a backslash, we'll append
			 * the following line to buf.  The backslash and newline
			 * will be overwritten due to 'n -= 2'.
			 */
			n -= 2;
			continue;
		}
		if (buf[n - 1] == '\n')
			buf[n - 1] = '\0';	/* delete newline */
		n = 0;
		/* skip whitespace at beginning of line */
		p = skip_whitespace(buf);
		if (*p == '\0' || *p == '#')
			continue;	/* skip empty line */
		/* okay, we have a line to parse, allocate memory */
		if ((current = malloc(sizeof(host))) == NULL) {
			error("cannot allocate memory: %m");
			if (list != NULL)
				free_hosts(list);
			return NULL;
		}
		/* read and save the hostname */
		len = token_length(p);
		if ((current->hostname = malloc(len + 1)) == NULL) {
			error("cannot allocate memory: %m");
			free(current);
			if (list != NULL)
				free_hosts(list);
			return NULL;
		}
		memcpy(current->hostname, p, len);
		current->hostname[len] = '\0';
		/* skip whitespace after hostname */
		p = skip_whitespace(p + len);
		/* is there a second column? */
		if (*p == '\0' || *p == '#') {
			error("no type and delay specified: %s:%d", file, line);
			free(current->hostname);
			free(current);
			if (list != NULL)
				free_hosts(list);
			return NULL;
		}
		/* read and save the type */
		len = token_length(p);
		if ((current->type = malloc(len + 1)) == NULL) {
			error("cannot allocate memory: %m");
			free(current->hostname);
			free(current);
			if (list != NULL)
				free_hosts(list);
			return NULL;
		}
		memcpy(current->type, p, len);
		current->type[len] = '\0';
		/* skip whitespace after type */
		p = skip_whitespace(p + len);
		/* is there a third column? */
		if (*p == '\0' || *p == '#') {
			error("no delay specified: %s:%d", file, line);
			free(current->hostname);
			free(current->type);
			free(current);
			if (list != NULL)
				free_hosts(list);
			return NULL;
		}
		/* read and save the delay */
		if (*p == '-') {
			error("delay value must be a positive integer: %s:%d",
			    file, line);
			free(current->hostname);
			free(current->type);
			free(current);
			if (list != NULL)
				free_hosts(list);
			return NULL;
		}
		current->delay = (unsigned long) strtol(p, &end, 10);
		if (p == end) {
			error("unexpected non-integer specified as delay: "
			    "%s:%d", file, line);
			free(current->hostname);
			free(current->type);
			free(current);
			if (list != NULL)
				free_hosts(list);
			return NULL;
		}
		/* skip trailing whitespace */
		p = skip_whitespace(end);
		if (*p != '#' && *p != '\0') {
			error("unexpected stuff ('%s') at end of line: %s:%d",
			    p, file, line);
			free(current->hostname);
			free(current->type);
			free(current);
			if (list != NULL)
				free_hosts(list);
			return NULL;
		}
		debug("saved %s %s %u", current->hostname, current->type,
		    current->delay);
		/* sort into list */
		for (prev = tmp = list; tmp != NULL &&
		    current->delay > tmp->delay; prev = tmp,
		    tmp = tmp->next)
			continue;
		if (tmp == list) {
			/* we're the first list element */
			list = current;
			list->next = tmp;
		} else {
			prev->next = current;
			current->next = tmp;
		}
	}
	if (ferror(fp) || !feof(fp)) {
		error("error while reading %s", file);
		if (list != NULL)
			free_hosts(list);
		return NULL;
	}
	if (fclose(fp) != 0) {
		error("cannot close %s: %m", file);
		if (list != NULL)
			free_hosts(list);
		return NULL;
	}
	if (list == NULL)
		warn("%s contains no entries", file);

	debug("done reading hosts file");
	return list;
}

void
free_hosts(host *list)
{
	host *s, *next;

	for (s = list; s != NULL; s = next) {
		next = s->next;
		free(s->hostname);
		free(s->type);
		free(s);
	}
	debug("free(3)d host list");
}

pid_t
start_shutdown(const general_conf *conf)
{
	host *current = NULL, *list;
	pid_t pid;
	time_t start, duration;
	unsigned int sleep_time;
	int status;

	block_signals(true);
	if ((pid = fork()) == -1) {
		error("cannot fork shutdown handling process: %m");
		block_signals(false);
		return pid;
	} else if (pid) {
		block_signals(false);
		return pid;	/* parent returns */
	}

	setup_shutdown_signals();
	block_signals(false);
	start = time(NULL);
	notice("initiating shutdown");

	if ((list = read_hosts(conf->hostsfile)) == NULL) {
		error("aborting shutdown");
		exit(EXIT_FAILURE);
	}

	for (current = list; current != NULL; current = current->next) {
		debug("next host to shutdown is %s", current->hostname);

		if (!shutdown_immediately) {
			/* check whether to sleep before shutdown */
			duration = time(NULL) - start;
			if (duration < (time_t) (current->delay * 60)) {
				sleep_time = current->delay * 60 - duration;
				info("shutting down %s in %u seconds",
				    current->hostname, sleep_time);
				while ((sleep_time = sleep(sleep_time)))
					if (shutdown_immediately)
						break;
			}
			/* check whether max_shutdowns is reached */
			if (conf->max_shutdowns)
				while (shutdowns_running >=
				    conf->max_shutdowns) {
					info("max_shutdowns (%d) reached, "
					    "waiting", conf->max_shutdowns);
					wait_for_shutdown_child();
					shutdowns_running--;
				}
		}

		/* shut down current->hostname */
		if (run_shutdown(conf->shutdown_cmd, current->hostname,
		    current->type) != -1)
			shutdowns_running++;
	}

	while (shutdowns_running) {
		debug("waiting for %d shutdowns", shutdowns_running);
		wait_for_shutdown_child();
		shutdowns_running--;
	}
	if (conf->suicide_cmd != NULL) {
		notice("remote shutdowns done, shutting down myself");
		if ((status = run(conf->suicide_cmd)) != 0)
			error("suicide command '%s' failed with status %d",
			    conf->suicide_cmd, status);
	}
	free_hosts(list);

	notice("shutdown process done");
	exit(EXIT_SUCCESS);
}

static pid_t
run_shutdown(const char *shutdown_cmd, const char *hostname, const char *type)
{
	pid_t pid;
	int status;
	char *tmp, *cmd;

	block_signals(true);
	if ((pid = fork()) == -1) {
		error("cannot fork shutdown handling process: %m");
		block_signals(false);
		return pid;
	} else if (pid) {
		block_signals(false);
		return pid;	/* parent returns */
	}
	setup_shutdown_child_signals();
	block_signals(false);

	if ((tmp = replace(shutdown_cmd, "%t", type)) == NULL)
		exit(EXIT_FAILURE);
	if ((cmd = replace(tmp, "%h", hostname)) == NULL)
		exit(EXIT_FAILURE);
	notice("shutting down %s", hostname);
	if ((status = run(cmd)) != 0) {
		error("shutdown command '%s' failed with status %d", cmd,
		    status);
		exit(EXIT_FAILURE);
	}
	free(tmp);
	free(cmd);
	exit(EXIT_SUCCESS);
}

static inline size_t
token_length(const char *buf)
{
	size_t length;

	for (length = 0; *buf != ' ' && *buf != '\t' && *buf != '\0'; buf++,
	    length++)
		continue;

	return length;
}

static void
wait_for_shutdown_child(void)
{
	pid_t pid;
	int status;

wait:
	if ((pid = waitpid(0, &status, 0)) > 0) {
		if (WIFEXITED(status))
			debug("shutdown child %d exited with code %d", pid,
			    WEXITSTATUS(status));
		else if (WIFSIGNALED(status))
			debug("shutdown child %d received %s", pid,
			    signal2str(WTERMSIG(status)));
		else
			error("shutdown child %d died for unknown reason", pid);
		return;
	}

	if (pid == -1 && errno == EINTR) {
		debug("waitpid(2) was interrupted");
		goto wait;
	}
	error("waitpid(2) returned %d: %m", pid);
}

static void
setup_shutdown_signals(void)
{
	/* TODO: use sigaction(2) (if available) */
	signal(SIGCHLD, SIG_DFL);
	signal(SIGPIPE, SIG_IGN);	/* ignore SIGPIPE */
	signal(SIGTSTP, SIG_IGN);	/* ignore TTY signals */
	signal(SIGTTOU, SIG_IGN);
	signal(SIGTTIN, SIG_IGN);
	signal(SIGHUP,  SIG_IGN);
	signal(SIGUSR1, shutdown_sighandler);
	signal(SIGUSR2, shutdown_sighandler);
	signal(SIGTERM, shutdown_sighandler);
	signal(SIGINT,  shutdown_sighandler);
}

static void
setup_shutdown_child_signals(void)
{
	/* TODO: use sigaction(2) (if available) */
	signal(SIGCHLD, SIG_DFL);
	signal(SIGPIPE, SIG_IGN);	/* ignore SIGPIPE */
	signal(SIGTSTP, SIG_IGN);	/* ignore TTY signals */
	signal(SIGTTOU, SIG_IGN);
	signal(SIGTTIN, SIG_IGN);
	signal(SIGHUP,  SIG_IGN);
	signal(SIGUSR1, shutdown_child_sighandler);
	signal(SIGUSR2, shutdown_child_sighandler);
	signal(SIGTERM, shutdown_child_sighandler);
	signal(SIGINT,  shutdown_child_sighandler);
}

RETSIGTYPE
shutdown_sighandler(int sig)
{
	setup_shutdown_signals();

	switch (sig) {
	case SIGUSR2:
		notice("received %s, shutting down hosts immediately",
		    signal2str(sig));
		shutdown_immediately = 1;
		break;
	/* terminating signals */
	case SIGINT:	/* FALLTHROUGH */
	case SIGUSR1:	/* FALLTHROUGH */
	case SIGTERM:
		if (!shutdown_termination_running) {
			shutdown_termination_running = 1;
			info("received %s", signal2str(sig));
			if (shutdowns_running) {
				info("waiting for %d currently running "
				    "shutdowns", shutdowns_running);
				while (shutdowns_running) {
					debug("waiting for %d shutdowns",
					    shutdowns_running);
					wait_for_shutdown_child();
					shutdowns_running--;
				}
			}
			notice("shutdown process terminating due to signal %s",
			    signal2str(sig));
			signal(sig, SIG_DFL);
			(void) raise(sig);	/* default action */
		}
		break;
	default:
		/* should never happen */
		error("signal %d not handled", sig);
	}
}

RETSIGTYPE
shutdown_child_sighandler(int sig)
{
	setup_shutdown_child_signals();

	/*
	 * For convenience, we'll forward handled signals to the shutdown
	 * parent, so that the user won't have to distinguish between the
	 * shutdown parent and the child processes.  Apart from that, we'll
	 * ignore SIGUSR2, since this tells us to immediately shutdown all
	 * hosts, which means that we must continue our shutdown.  For all other
	 * handled signals, we'll apply the default action to ourselves, which
	 * should be termination.
	 */
	(void) kill(getppid(), sig);
	if (sig != SIGUSR2) {
		info("received %s, terminating", signal2str(sig));
		signal(sig, SIG_DFL);
		(void) raise(sig);	/* default action */
	}
}
