/*
 * $Id$
 *
 * Copyright (c) 2005, 2006, 2011 Freie Universitaet Berlin.
 * All rights reserved.
 *
 * Written by Holger Weiss <holger@ZEDAT.FU-Berlin.DE>.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif			/* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "log.h"
#include "snmpget.h"
#include "zupsd.h"

/* <net-snmp/net-snmp-config.h> redefines these: */
#undef PACKAGE_BUGREPORT
#undef PACKAGE_NAME
#undef PACKAGE_STRING
#undef PACKAGE_TARNAME
#undef PACKAGE_VERSION

#include <net-snmp/net-snmp-config.h>
#include <net-snmp/net-snmp-includes.h>

#define DEFAULT_SNMP_RETRIES       5
#define DEFAULT_SNMP_TIMEOUT 1000000

RCSID("$Id$");

extern int snmp_errno;

/*
 * Sets request->value for each request->variable.  Supports SNMPv1
 * integer/string types only.  Returns true on success, false on error.
 */
bool
snmp_get_variables(const char *agent, const char *community, int retries,
                   long timeout, snmp_variables *request, char *errstr)
{
	struct snmp_session session, *s;
	struct snmp_pdu *pdu, *response = NULL;
	struct variable_list *rv;
	snmp_variables *var;
	oid objid[MAX_OID_LEN];
	size_t objid_len = MAX_OID_LEN;
	char o1[512], o2[512];
	int status, cliberr, snmperr;
	char *snmpagent, *snmpcommunity, *errbuf;

	if ((snmpagent = strdup(agent)) == NULL) {
		(void) strncpy(errstr, "cannot allocate memory",
		    MAXERRSTRLEN - 1);
		errstr[MAXERRSTRLEN - 1] = '\0';
		return false;
	}
	if ((snmpcommunity = strdup(community)) == NULL) {
		(void) strncpy(errstr, "cannot allocate memory",
		    MAXERRSTRLEN - 1);
		errstr[MAXERRSTRLEN - 1] = '\0';
		free(snmpagent);
		return false;
	}

	/* initialize SNMP library and session */
	init_snmp("zupsd");
	snmp_sess_init(&session);

	/* configure session */
	session.version = SNMP_VERSION_1;
	session.peername = snmpagent;
	session.community = (unsigned char *) community;
	session.community_len = strlen((char *) session.community);
	session.timeout = (timeout >= 0) ? timeout : DEFAULT_SNMP_TIMEOUT;
	session.retries = (retries >= 0) ? retries : DEFAULT_SNMP_RETRIES;

	/* open session */
	SOCK_STARTUP;
	if ((s = snmp_open(&session)) == NULL) {
		snmp_error(&session, &cliberr, &snmperr, &errbuf);
		(void) snprintf(errstr, MAXERRSTRLEN,
		    "cannot open SNMP session: %s", errbuf);
		free(errbuf);
		free(snmpagent);
		free(snmpcommunity);
		return false;
	}
	/* process all request variables */
	for (var = request; var != NULL; var = var->next) {
		/* create the PDU for the data of our request */
		if ((pdu = snmp_pdu_create(SNMP_MSG_GET)) == NULL) {
			(void) strncpy(errstr, "cannot create SNMP PDU",
			    MAXERRSTRLEN - 1);
			errstr[MAXERRSTRLEN - 1] = '\0';
			free(snmpagent);
			free(snmpcommunity);
			return false;
		}
		/* translate var->variable to an OID type */
		if (!read_objid(var->variable, objid, &objid_len)) {
			(void) snprintf(errstr, MAXERRSTRLEN,
			    "cannot parse SNMP OID: %s",
			    snmp_api_errstring(snmp_errno));
			free(snmpagent);
			free(snmpcommunity);
			return false;
		}
		/* add the OID to the PDU */
		(void) snmp_add_null_var(pdu, objid, objid_len);
		/* send the request out */
		status = snmp_synch_response(s, pdu, &response);
		/* process the response */
		if (status != STAT_SUCCESS) {
			if (status == STAT_TIMEOUT) {
				(void) strncpy(errstr,
				    "SNMP timeout: no response from agent",
				    MAXERRSTRLEN - 1);
				errstr[MAXERRSTRLEN - 1] = '\0';
			} else {
				snmp_sess_error(s, &cliberr, &snmperr, &errbuf);
				(void) snprintf(errstr, MAXERRSTRLEN,
				    "SNMP session error: %s", errbuf);
				free(errbuf);
			}
			free(snmpagent);
			free(snmpcommunity);
			return false;
		} else if (response->errstat != SNMP_ERR_NOERROR) {
			(void) snprintf(errstr, MAXERRSTRLEN, "SNMP error: %s",
			    snmp_errstring(response->errstat));
			free(snmpagent);
			free(snmpcommunity);
			return false;
		} else {
			rv = response->variables;
			/* make sure the response is what we requested */
			if (snprint_objid(o1, sizeof(o1), objid, objid_len)
			    == -1) {
				(void) strncpy(errstr,
				    "buffer too small for SNMP request OID",
				    MAXERRSTRLEN - 1);
				errstr[MAXERRSTRLEN - 1] = '\0';
				free(snmpagent);
				free(snmpcommunity);
				return false;
			}
			if (snprint_objid(o2, sizeof(o2), rv->name,
			    rv->name_length) == -1) {
				(void) strncpy(errstr,
				    "buffer too small for SNMP response OID",
				    MAXERRSTRLEN - 1);
				errstr[MAXERRSTRLEN - 1] = '\0';
				free(snmpagent);
				free(snmpcommunity);
				return false;
			}
			/* TODO: write an "oidcmp(oid *o1, oid *o2)" */
			if (strcmp(o1, o2)) {
				(void) snprintf(errstr, MAXERRSTRLEN,
				    "SNMP response OID %s doesn't match "
				    "request OID %s", o2, o1);
				free(snmpagent);
				free(snmpcommunity);
				return false;
			}
			/* we only requested one single variable */
			if (rv->next_variable != NULL) {
				(void) strncpy(errstr,
				    "more variables returned than requested",
				    MAXERRSTRLEN - 1);
				free(snmpagent);
				free(snmpcommunity);
				return false;
			}
			/* the response seems to be okay */
			if (rv->type == ASN_OCTET_STR) {
				if (var->type != SNMP_STR_TYPE) {
					(void) snprintf(errstr, MAXERRSTRLEN,
					    "string type returned but not "
					    "expected for %s", var->variable);
					free(snmpagent);
					free(snmpcommunity);
					return false;
				}
				/* save string */
				if ((var->value.string =
				    malloc(rv->val_len + 1)) == NULL) {
					(void) snprintf(errstr, MAXERRSTRLEN,
					    "cannot allocate memory for %s",
					    var->variable);
					free(snmpagent);
					free(snmpcommunity);
					return false;
				}
				memcpy(var->value.string, rv->val.string,
				    rv->val_len);
				var->value.string[rv->val_len] = '\0';
			} else if (rv->type == ASN_INTEGER) {
				if (var->type != SNMP_INT_TYPE) {
					(void) snprintf(errstr, MAXERRSTRLEN,
					    "integer type returned but not "
					    "expected for %s", var->variable);
					free(snmpagent);
					free(snmpcommunity);
					return false;
				}
				/* save integer */
				var->value.integer = *rv->val.integer;
			} else {
				(void) snprintf(errstr, MAXERRSTRLEN,
				    "unexpected type returned for %s",
				    var->variable);
				free(snmpagent);
				free(snmpcommunity);
				return false;
			}
		}
		if (response != NULL)
			snmp_free_pdu(response);
	}
	/* close session and clean up */
	if (!snmp_close(s)) {
		snmp_sess_error(s, &cliberr, &snmperr, &errbuf);
		(void) snprintf(errstr, MAXERRSTRLEN,
		    "cannot close SNMP session: %s", errbuf);
		free(errbuf);
		free(snmpagent);
		free(snmpcommunity);
		return false;
	}
	SOCK_CLEANUP;
	free(snmpagent);
	free(snmpcommunity);

	debug("SNMP session with %s successful", agent);
	return true;
}
