/*
 * $Id$
 *
 * Copyright (c) 2005, 2006, 2011 Freie Universitaet Berlin.
 * All rights reserved.
 *
 * Written by Holger Weiss <holger@ZEDAT.FU-Berlin.DE>.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif			/* HAVE_CONFIG_H */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "conf.h"
#include "log.h"
#include "notify.h"
#include "ups.h"
#include "snmpget.h"
#include "zupsd.h"

RCSID("$Id$");

#define MAXMESSAGELEN 8192

/*
 * IETF UPS MIB objects (iso.org.dod.internet.mgmt.mib.upsMIB.ups)
 */
#define IETF                                "1.3.6.1.2.1.33.1."
#define upsIdentManufacturer                IETF "1.1.0"
#define upsIdentModel                       IETF "1.2.0"
#define upsIdentUPSSoftwareVersion          IETF "1.3.0"
#define upsIdentAgentSoftwareVersion        IETF "1.4.0"
#define upsBatterySecondsOnBattery          IETF "2.2.0"
#define upsBatteryEstimatedMinutesRemaining IETF "2.3.0"
#define upsBatteryEstimatedChargeRemaining  IETF "2.4.0"
#define upsBatteryVoltage                   IETF "2.5.0"
#define upsOutputSource                     IETF "4.1.0"

#define INT SNMP_INT_TYPE
#define STR SNMP_STR_TYPE

#define ADD_SNMP_VAR(p, v, t)                                                  \
	do {                                                                   \
		snmp_variables *new, *tmp;                                     \
                                                                               \
		if ((new = malloc(sizeof(snmp_variables))) == NULL) {          \
			error("cannot allocate memory: %m");                   \
			return MODULE_UNKNOWN;                                 \
		}                                                              \
		if (p == NULL)                                                 \
			p = new;                                               \
		else {                                                         \
			for (tmp = p; tmp->next != NULL; tmp = tmp->next)      \
				continue;                                      \
			tmp->next = new;                                       \
		}                                                              \
		new->variable = (char *) v;                                    \
		if (t == INT)                                                  \
			new->type = INT;                                       \
		else {                                                         \
			new->type = STR;                                       \
			new->value.string = NULL;                              \
		}                                                              \
		new->next = NULL;                                              \
	} while (/* CONSTCOND */ 0)

#define FREE_SNMP_VARS(p)                                                      \
	do {                                                                   \
		snmp_variables *tmp, *next;                                    \
                                                                               \
		for (tmp = p; tmp != NULL; tmp = next) {                       \
			if (tmp->type == STR &&                                \
			    tmp->value.string != NULL)                         \
				free(tmp->value.string);                       \
			next = tmp->next;                                      \
			free(tmp);                                             \
		}                                                              \
		p = NULL;                                                      \
	} while (/* CONSTCOND */ 0)

/* TODO: We should get rid of this string comparison */
#define IS_SNMP_VAR(p) (!strcmp(response->variable, p))

int
ups_test(void *config)
{
	ups_conf *ups = (ups_conf *) config;
	snmp_variables *request = NULL, *response;
	char errstr[MAXERRSTRLEN];
	char *tmp, *hostname;
	bool snmp_error = false;

	ADD_SNMP_VAR(request, upsIdentManufacturer, STR);
	ADD_SNMP_VAR(request, upsIdentModel, STR);
	ADD_SNMP_VAR(request, upsIdentUPSSoftwareVersion, STR);
	ADD_SNMP_VAR(request, upsIdentAgentSoftwareVersion, STR);

	if ((tmp = strdup(ups->hostname)) == NULL) {
		error("cannot allocate memory: %m");
		return MODULE_UNKNOWN;
	}
	for (hostname = strtok(tmp, ", "); hostname != NULL;
	    hostname = strtok(NULL, ", ")) {
		info("UPS module connecting to %s", hostname);

		if (snmp_get_variables(hostname, ups->community,
		    (int) ups->retries, ups->timeout * 1000, request, errstr))
			snmp_error = false;
		else {
			snmp_error = true;
			continue;
		}

		for (response = request; response != NULL;
		    response = response->next) {
			if (IS_SNMP_VAR(upsIdentManufacturer))
				info("%s: Manufacturer: %s", hostname,
				    response->value.string);
			else if (IS_SNMP_VAR(upsIdentModel))
				info("%s: Model: %s", hostname,
				    response->value.string);
			else if (IS_SNMP_VAR(upsIdentUPSSoftwareVersion))
				info("%s: Firmware: %s", hostname,
				    response->value.string);
			else if (IS_SNMP_VAR(upsIdentAgentSoftwareVersion))
				info("%s: SNMP agent: %s", hostname,
				    response->value.string);
			else {
				/* should never happen */
				error("%s: unexpected response variable %s",
				    hostname, response->variable);
				free(tmp);
				FREE_SNMP_VARS(request);
				return MODULE_UNKNOWN;
			}
		}
		break;
	}
	free(tmp);
	FREE_SNMP_VARS(request);

	if (snmp_error) {
		error("%s: %s", ups->hostname, errstr);
		return MODULE_UNKNOWN;
	}
	return ups_check(config);
}

int
ups_check(void *config)
{
#define NOTIFY_THRESHOLD 5
	ups_conf *ups = (ups_conf *) config;
	static snmp_variables *request = NULL;
	snmp_variables *response;
	int state = MODULE_UNKNOWN;
	char message[MAXMESSAGELEN], errstr[MAXERRSTRLEN];
	char *tmp, *hostname;
	bool snmp_error = false;

	if ((tmp = strdup(ups->hostname)) == NULL) {
		error("cannot allocate memory: %m");
		return MODULE_UNKNOWN;
	}
	for (hostname = strtok(tmp, ", "); hostname != NULL;
	    hostname = strtok(NULL, ", ")) {
		if (request == NULL) {
			ADD_SNMP_VAR(request,
			    upsBatteryEstimatedMinutesRemaining, INT);
			ADD_SNMP_VAR(request,
			    upsBatteryEstimatedChargeRemaining, INT);
			ADD_SNMP_VAR(request, upsBatterySecondsOnBattery, INT);
			ADD_SNMP_VAR(request, upsBatteryVoltage, INT);
			ADD_SNMP_VAR(request, upsOutputSource, INT);
		}

		if (snmp_get_variables(hostname, ups->community,
		    (int) ups->retries, ups->timeout * 1000, request, errstr))
			snmp_error = false;
		else {
			snmp_error = true;
			continue;
		}

		(void) strncpy(errstr, "unknown error", sizeof(errstr) - 1);
		errstr[sizeof(errstr) - 1] = '\0';

		for (response = request; response != NULL;
		    response = response->next) {
			if (IS_SNMP_VAR(upsOutputSource)) {
				switch (response->value.integer) {
				case 1:
					notice("%s: output source: other",
					    hostname);
					(void) strncpy(errstr,
					    "output source: other",
					    sizeof(errstr) - 1);
					errstr[sizeof(errstr) - 1] = '\0';
					state = MODULE_WARNING;
					break;
				case 2:
					notice("%s: output source: none",
					    hostname);
					(void) strncpy(errstr,
					    "output source: none",
					    sizeof(errstr) - 1);
					errstr[sizeof(errstr) - 1] = '\0';
					state = MODULE_WARNING;
					break;
				case 3:
					info("%s: output source: normal",
					    hostname);
					state = MODULE_OK;
					break;
				case 4:
					notice("%s: output source: bypass",
					    hostname);
					(void) strncpy(errstr,
					    "output source: bypass",
					    sizeof(errstr) - 1);
					errstr[sizeof(errstr) - 1] = '\0';
					state = MODULE_WARNING;
					break;
				case 5:
					warn("%s: output source: battery",
					    hostname);
					/* don't set CRITICAL during an ALERT */
					if (state < MODULE_CRITICAL) {
						(void) strncpy(errstr,
						    "output source: battery",
						    sizeof(errstr) - 1);
						errstr[sizeof(errstr) - 1]
						    = '\0';
						state = MODULE_CRITICAL;
					}
					break;
				case 6:
					notice("%s: output source: booster",
					    hostname);
					(void) strncpy(errstr,
					    "output source: booster",
					    sizeof(errstr) - 1);
					errstr[sizeof(errstr) - 1] = '\0';
					state = MODULE_WARNING;
					break;
				case 7:
					notice("%s: output source: reducer",
					    hostname);
					(void) strncpy(errstr,
					    "output source: reducer",
					    sizeof(errstr) - 1);
					errstr[sizeof(errstr) - 1] = '\0';
					state = MODULE_WARNING;
					break;
				default:
					error("%s: unknown upsOutputSource "
					    "value: %ld",
					    response->value.integer);
					(void) snprintf(errstr, sizeof(errstr),
					    "unknown upsOutputSource value: "
					    "%ld", response->value.integer);
					state = MODULE_UNKNOWN;
					goto notify;
				}
			} else if (
			    IS_SNMP_VAR(upsBatteryEstimatedMinutesRemaining)) {
				/*
				 * We'll go into ALERT state only if the UPS is
				 * on battery.  Therefore, if an OK or WARNING
				 * state was set before, we're not going to set
				 * ALERT.  If we set ALERT and a non-CRITICAL
				 * state is encountered later, our ALERT will be
				 * overridden.
				 *
				 * TODO: Seperate the setting of our module
				 * state from this variable value parsing loop.
				 */
				if (state != MODULE_OK
				    && state != MODULE_WARNING
				    && ups->alert_threshold
				    && response->value.integer
				    <= ups->alert_threshold) {
					warn("%s: %ld minutes battery "
					    "remaining", hostname,
					    response->value.integer);
					(void) snprintf(errstr, sizeof(errstr),
					    "%ld minutes battery remaining",
					    response->value.integer);
					state = MODULE_ALERT;
				} else
					info("%s: %ld minutes battery "
					    "remaining", hostname,
					    response->value.integer);
			/*
			 * The following variables are not used for setting our
			 * module state, the values are logged only.
			 */
			} else if (IS_SNMP_VAR(upsBatterySecondsOnBattery)) {
				if (response->value.integer)
					notice("%s: %ld seconds on battery",
					    hostname, response->value.integer);
			} else if (
			    IS_SNMP_VAR(upsBatteryEstimatedChargeRemaining)) {
				info("%s: %ld%% charge level", hostname,
				    response->value.integer);
			} else if (IS_SNMP_VAR(upsBatteryVoltage)) {
				info("%s: battery voltage: %lddV", hostname,
				    response->value.integer);
			} else {
				/* should never happen */
				error("%s: internal error: unexpected SNMP "
				    "variable returned instead of %s", hostname,
				    response->variable);
				(void) snprintf(errstr, sizeof(errstr),
				    "unexpected SNMP variable returned instead "
				    "of %s", response->variable);
				state = MODULE_UNKNOWN;
				goto notify;
			}
		}
		break;
	}

	if (snmp_error)
		error("%s: %s", ups->hostname, errstr);

notify:
	if (state > ups->notified_by_email && ups->email_infolevel + state >=
	    NOTIFY_THRESHOLD && ups->email_cmd != NULL) {
		(void) snprintf(message, sizeof(message), "%s: %s - %s",
		    ups->hostname, state2str(state), errstr);
		(void) notify(ups->email_cmd, ups->email_contact, message);
		ups->notified_by_email = state;
	}
	if (state > ups->notified_by_pager && ups->pager_infolevel + state >=
	    NOTIFY_THRESHOLD && ups->pager_cmd != NULL) {
		(void) snprintf(message, sizeof(message), "%s: %s - %s",
		    ups->hostname, state2str(state), errstr);
		(void) notify(ups->pager_cmd, ups->pager_contact, message);
		ups->notified_by_pager = state;
	}
	if (state == MODULE_OK) {
		if (ups->notified_by_email) {
			(void) snprintf(message, sizeof(message), "%s: %s - "
			    "recovery", ups->hostname, state2str(state));
			(void) notify(ups->email_cmd, ups->email_contact,
			    message);
			ups->notified_by_email = 0;
		}
		if (ups->notified_by_pager) {
			(void) snprintf(message, sizeof(message), "%s: %s - "
			    "recovery", ups->hostname, state2str(state));
			(void) notify(ups->pager_cmd, ups->pager_contact,
			    message);
			ups->notified_by_pager = 0;
		}
	}

	return state;
}
