/*
 * $Id$
 *
 * Copyright (c) 2005, 2006, 2009 Freie Universitaet Berlin.
 * All rights reserved.
 *
 * Written by Holger Weiss <holger@ZEDAT.FU-Berlin.DE>.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif			/* HAVE_CONFIG_H */

#if HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif			/* HAVE_SYS_IOCTL_H */
#if HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif			/* HAVE_SYS_TYPES_H */
#if HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif			/* HAVE_SYS_STAT_H */
#if HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif			/* HAVE_SYS_WAIT_H */
#include <errno.h>
#if HAVE_FCNTL_H
#include <fcntl.h>
#endif			/* HAVE_FCNTL_H */
#if HAVE_LIBGEN_H
#include <libgen.h>
#endif			/* HAVE_LIBGEN_H */
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#if HAVE_UNISTD_H
#include <unistd.h>
#endif			/* HAVE_UNISTD_H */

#include "conf.h"
#include "log.h"
#include "shutdown.h"
#include "sig.h"
#include "zupsd.h"

RCSID("$Id$");

/* command line flags */
#define FOREGROUND_FLAG 0x01
#define PIDFILE_FLAG    0x02
#define QUIET_FLAG      0x04
#define TEST_FLAG       0x08

/*
 * The volatile modifier can safely be casted away when handing the conf pointer
 * over to the fork(2)ed shutdown process, and when using the conffile or the
 * pidfile pointer (which are read-only after initialization).
 */
#undef PIDFILE	/* defined in config.h */
#define PIDFILE (const char *) pidfile
#define CONFFILE (const char *) conffile
#define CONF (general_conf *) conf

enum shutdown_states {
	INACTIVE,
	ACTIVE,
	ALERT,
	USER,
	DONE
};

extern int errno;
extern int optind;
extern int optopt;
extern int opterr;
extern int optreset;
extern char *optarg;

int debugging = 0;

/* static volatile variables used in sighandler */
static volatile sig_atomic_t shutdown_state = INACTIVE;
static volatile sig_atomic_t termination_running = 0;
static volatile module *new_modules;
static volatile general_conf *conf;
static volatile pid_t shutdown_pid = 0;
static volatile int pidfd = -1;
static volatile const char *conffile;
static volatile const char *pidfile;

static int test_run(module *modules);
static void myself(FILE *stream);
static void usage(FILE *stream, const char *progname);
static void version(void);
static void stop_shutdown(void);
static bool reap_dead_child(volatile pid_t *spid, bool block, bool suicide);
static bool detach(void);
static bool write_pidfile(const char *file);
static bool setup_process(void);
static void setup_descriptors(void);
static void setup_signals(void);
RETSIGTYPE sighandler(int sig);

int
main(int argc, char **argv)
{
	struct {
		int ok;
		int warn;
		int crit;
		int alrt;
		int unkn;
	} mstates;
	module *mod, *modules;
	host *list;
	int i, output, facility, flags = 0;
	unsigned int sleep_time;
	time_t start, duration;
	const char *progname, *optconffile = NULL, *optpidfile = NULL;

#if HAVE_BASENAME
	progname = basename(argv[0]);
#else
	progname = "zupsd";
#endif
	while ((i = getopt(argc, argv, "Ddf:hPp:qtV")) != -1)
		switch (i) {
		case 'D':
			flags |= FOREGROUND_FLAG;
			break;
		case 'd':
			debugging++;
			break;
		case 'f':
			optconffile = optarg;
			break;
		case 'h':
			usage(stdout, progname);
			exit(EXIT_SUCCESS);
			/* NOTREACHED */
		case 'P':
			flags |= PIDFILE_FLAG;
			break;
		case 'p':
			optpidfile = optarg;
			break;
		case 'q':
			flags |= QUIET_FLAG;
			break;
		case 't':
			flags |= TEST_FLAG;
			break;
		case 'V':
			version();
			exit(EXIT_SUCCESS);
			/* NOTREACHED */
		case '?':
			/* FALLTHROUGH */
		default:
			usage(stderr, progname);
			exit(EXIT_FAILURE);
		}

	if (argc - optind) {
		(void) fprintf(stderr, "%s: cannot parse argument -- %s\n",
		    progname, argv[optind]);
		usage(stderr, progname);
		exit(EXIT_FAILURE);
	}

	if (flags & QUIET_FLAG)
		output = ZUPSD_SYSLOG;
	else if (flags & TEST_FLAG || flags & FOREGROUND_FLAG)
		output = ZUPSD_STDERR;
	else
		output = ZUPSD_SYSLOG | ZUPSD_STDERR;

	open_log(progname, str2facility(DEFAULT_FACILITY_NAME), output);

	/* read configuration file */
	conffile = (optconffile != NULL) ? optconffile : DEFAULT_CONFFILE;
	if ((new_modules = modules = readconf(CONFFILE)) == NULL) {
		error("error reading configuration file %s, exiting", conffile);
		exit(EXIT_FAILURE);
	}

	/* save conf static for sighandler */
	conf = (general_conf *) modules->config;
	/* configured syslog facility */
	facility = str2facility(conf->syslog_facility);
	/* reopen syslog with the configured facility */
	if (output & ZUPSD_SYSLOG &&
	    strcmp(conf->syslog_facility, DEFAULT_FACILITY_NAME))
		reopen_log(progname, facility, output);

	/* command line overwrites conf */
	if (conf->debug_level && !debugging)
		debugging = conf->debug_level;

	/* again, command line overwrites conf */
	pidfile = (optpidfile != NULL) ? optpidfile : conf->pidfile;

	/* print pidfile and exit */
	if (flags & PIDFILE_FLAG) {
		(void) puts(PIDFILE);
		exit(EXIT_SUCCESS);
	}
	/* do test run and exit */
	if (flags & TEST_FLAG) {
		if ((list = read_hosts(conf->hostsfile)) == NULL)
			exit(EXIT_FAILURE);
		free_hosts(list);
		info("hosts file looks fine");
		exit(test_run(modules->next));
	}

	/* detach unless foreground run */
	if (!(flags & FOREGROUND_FLAG)) {
		/* detach from TTY */
		if (!detach()) {
			error("cannot daemonize, exiting");
			exit(EXIT_FAILURE);
		}
		/* close descriptors and reopen syslog */
		close_log();
		setup_descriptors();
		open_log(progname, facility, ZUPSD_SYSLOG);
	}
	/* write pidfile */
	if (!write_pidfile(PIDFILE)) {
		error("cannot write pidfile, exiting");
		exit(EXIT_FAILURE);
	}
	/* setup umask, work directory, and signals */
	if (!setup_process()) {
		error("cannot setup process, exiting");
		exit(EXIT_FAILURE);
	}

	/* main daemon loop */
	while (1) {
		start = time(NULL);
		(void) memset(&mstates, 0, sizeof(mstates));

		/* we're paranoid and won't rely on SIGCHLD delivery */
		if (shutdown_pid && reap_dead_child(&shutdown_pid, false,
		    conf->commit_suicide) && shutdown_state != INACTIVE)
			/*
			 * In case of recovery or SIGHUP, shutdown_state was set
			 * to INACTIVE prior to killing the child.  In all other
			 * cases (that is, if either the shutdown was completed
			 * or the user terminated the process), we'll set
			 * shutdown_state to DONE.
			 */
			shutdown_state = DONE;

		/* check whether the configuration was reread */
		if (modules != new_modules) {
			mod = modules;
			modules = (module *) new_modules;
			conf = (general_conf *) modules->config;
			free_modules(mod);
			notice("new configuration will be used");
		}

		/* run module checks (skip "general" module) */
		for (mod = modules->next; mod != NULL; mod = mod->next)
			switch ((mod->check)(mod->config)) {
			case MODULE_OK:
				mstates.ok++;
				break;
			case MODULE_UNKNOWN:
				mstates.unkn++;
				break;
			case MODULE_WARNING:
				mstates.warn++;
				break;
			case MODULE_CRITICAL:
				mstates.crit++;
				break;
			case MODULE_ALERT:
				mstates.alrt++;
				break;
			default:
				/* should never happen */
				error("internal error: module returned unknown "
				    "status");
				mstates.unkn++;
			}

		/* log results */
		if (mstates.ok)
			info("%d module(s) returned OK", mstates.ok);
		if (mstates.unkn)
			warn("%d module(s) returned UNKNOWN", mstates.unkn);
		if (mstates.warn)
			warn("%d module(s) returned WARNING", mstates.warn);
		if (mstates.crit)
			warn("%d module(s) returned CRITICAL", mstates.crit);
		if (mstates.alrt)
			warn("%d module(s) returned ALERT", mstates.alrt);

		/* check whether some action is required */
		if (mstates.alrt >= conf->num_alert) {
			if (shutdown_state == INACTIVE) {
				if ((shutdown_pid = start_shutdown(CONF)) != -1)
					shutdown_state = ACTIVE;
				else
					shutdown_pid = 0;
			}
			if (shutdown_pid && shutdown_state != ALERT) {
				if (kill(shutdown_pid, SIGUSR2) == -1)
					error("cannot send SIGUSR2 signal to "
					    "shutdown process %d: %m",
					    shutdown_pid);
				else
					shutdown_state = ALERT;
			}
		} else if (mstates.crit + mstates.alrt >= conf->num_critical) {
			if (shutdown_state == INACTIVE) {
				if ((shutdown_pid = start_shutdown(CONF)) != -1)
					shutdown_state = ACTIVE;
				else
					shutdown_pid = 0;
			}
		} else if (shutdown_state == ACTIVE || shutdown_state == ALERT) {
			notice("module recovery, aborting shutdown");
			shutdown_state = INACTIVE;
			stop_shutdown();
		} else if (shutdown_state == DONE) {
			notice("module recovery after shutdown");
			shutdown_state = INACTIVE;
		}

		/* go to sleep */
		duration = time(NULL) - start;
		if (duration < conf->check_interval) {
			sleep_time = conf->check_interval - duration;
			while ((sleep_time = sleep(sleep_time)))
				continue;
		}
	}

	return 0;	/* NOTREACHED */
}

static int
test_run(module *modules)
{
	module *mod;
	bool failed = false;

	info("testing configured modules");
	for (mod = modules; mod != NULL; mod = mod->next)
		if ((mod->test)(mod->config) != MODULE_OK)
			failed = true;

	if (failed) {
		warn("one or more module test(s) failed");
		return EXIT_FAILURE;
	}

	info("configuration seems to work fine");
	return EXIT_SUCCESS;
}

static void
myself(FILE *stream)
{
	(void) fprintf(stream, "%s (%s)\n", ZUPSD_STRING, RELEASE_DATE);
}

static void
usage(FILE *stream, const char *progname)
{
	myself(stream);
	(void) fprintf(stream,
	"Usage: %s [-DPqtd[d]] [-f file]\n"
	"       %s -h | -V\n"
	"Options:\n"
	"  -D      Run main zupsd process in foreground.\n"
	"  -d      Generate debug messages (-dd is more verbose).\n"
	"  -f file Read zupsd configuration from file.\n"
	"  -h      Print this help message and exit.\n"
	"  -P      Print zupsd pidfile path and exit.\n"
	"  -p file Write zupsd process ID to file.\n"
	"  -q      Never write messages to standard error output.\n"
	"  -t      Test current configuration and exit.\n"
	"  -V      Print zupsd version and exit.\n",
	progname, progname);
}

static void
version(void)
{
	myself(stdout);
	(void) puts(COPYRIGHT);
}

static void
stop_shutdown(void)
{
	if (shutdown_pid) {
		debug("killing shutdown process %d", shutdown_pid);
		if (kill(shutdown_pid, SIGTERM) == -1)
			error("cannot kill shutdown process %d: %m",
			    shutdown_pid);
	}
}

static bool
reap_dead_child(volatile pid_t *spid, bool block, bool suicide)
{
	pid_t pid;
	int status, options = 0;
	bool reaped_target = false;

	if (!block)
		options |= WNOHANG;
wait:
	while ((pid = waitpid(0, &status, options)) > 0)
		/*
		 * We'll reap all dead children, but we're mainly interested
		 * in spid (that is, in the shutdown process).
		 */
		if (pid == *spid) {
			*spid = 0;
			reaped_target = true;
			if (WIFEXITED(status))
				debug("shutdown process %d exited with code %d",
				    pid, WEXITSTATUS(status));
			else if (WIFSIGNALED(status))
				debug("shutdown process %d received %s", pid,
				    signal2str(WTERMSIG(status)));
			else
				error("shutdown process %d died for unknown "
				    "reason", pid);
			/* TODO: let the caller do the suicide */
			if (suicide) {
				notice("shutdown process done, killing myself");
				(void) raise(SIGTERM);
			}
			/* don't block while waiting for other childs */
			if (block) {
				options |= WNOHANG;
				block = false;
			}
		} else
			debug("child process %d exited", pid);

	if (pid == -1 && errno == EINTR) {
		debug("waitpid(2) was interrupted");
		if (block && !*spid) {
			options |= WNOHANG;
			block = false;
		}
		goto wait;
	}
	return reaped_target;
}

static bool
detach(void)
{
#if !HAVE_SETSID
#if !SETPGRP_VOID
	int fd;
#else
	pid_t pid;
#endif			/* !SETPGRP_VOID */
#endif			/* !HAVE_SETSID */

	if (getppid() == 1)	/* called via inittab or daemon already */
		return true;
	switch (fork()) {
	case -1:
		error("cannot fork(2) daemon process: %m");
		return false;
		/* NOTREACHED */
	case 0:	/* child becomes a daemon */
#if HAVE_SETSID
		/* POSIX */
		if (setsid() == -1) {
			error("setsid(2) failed for daemon process: %m");
			return false;
		}
#elif SETPGRP_VOID
		/* System V */
		if (setpgrp() == -1) {
			error("setpgrp(2) failed for daemon process: %m");
			return false;
		}
		signal(SIGCHLD, SIG_IGN);
		signal(SIGHUP, SIG_IGN);
		if ((pid = fork()) == -1) {
			error("cannot fork(2) daemon process: %m");
			return false;
		} else if (pid) /* parent exits */
			exit(EXIT_SUCCESS);
#else
		/* BSD */
		if (setpgrp(0, getpid()) == -1) {
			error("setpgrp(2) failed for daemon process: %m");
			return false;
		}
		if ((fd = open("/dev/tty", O_RDWR)) >= 0) {
			(void) ioctl(fd, TIOCNOTTY, (char *) 0);
			(void) close(fd);
		}
#endif			/* HAVE_SETSID */
		break;
	default:	/* parent exits */
		exit(EXIT_SUCCESS);
	}
	return true;
}

static bool
write_pidfile(const char *file)
{
	char pid[16];

	(void) snprintf(pid, sizeof(pid), "%d\n", getpid());
	if ((pidfd = open(file, O_RDWR | O_CREAT, 0640)) == -1) {
		error("opening %s failed: %m", file);
		return false;
	}
#if HAVE_LOCKF
	/*
	 * Locking the pidfile is not critical for us, so simply omit that if
	 * the system does not provide lockf(3).
	 */
	if (lockf(pidfd, F_TLOCK, 0) == -1) {
		error("locking %s failed, maybe another instance running: %m",
		    file);
		return false;
	}
#endif
	if (write(pidfd, pid, strlen(pid)) == -1) {
		error("writing pid to %s failed: %m", file);
		return false;
	}
	debug("PID %d written to pidfile %s", getpid(), file);
	return true;
}

static bool
setup_process(void)
{
#if HAVE_UMASK
	(void) umask(022);
#endif
	if (chdir("/") == -1) {
		error("chdir(2) to root directory failed: %m");
		return false;
	}
	setup_signals();
	return true;
}

static void
setup_descriptors(void)
{
	int fd;

	/* close all descriptors */
#if HAVE_GETDTABLESIZE
	fd = getdtablesize();
#else
	fd = 19;
#endif
	for (; fd >= 0; --fd)
		(void) close(fd);
	/* reopen stdin on /dev/null */
	fd = open("/dev/null", O_RDWR);
	/* reopen stdout */
	(void) dup(fd);
	/* reopen stderr */
	(void) dup(fd);
}

static void
setup_signals(void)
{
	/* TODO: use sigaction(2) (if available) */
	signal(SIGPIPE, SIG_IGN);	/* ignore SIGPIPE */
	signal(SIGTSTP, SIG_IGN);	/* ignore TTY signals */
	signal(SIGTTOU, SIG_IGN);
	signal(SIGTTIN, SIG_IGN);
	signal(SIGCHLD, sighandler);
	signal(SIGUSR1, sighandler);
	signal(SIGUSR2, sighandler);
	signal(SIGTERM, sighandler);
	signal(SIGINT,  sighandler);
	signal(SIGHUP,  sighandler);
}

RETSIGTYPE
sighandler(int sig)
{
	module *tmp_modules;

	setup_signals();

	switch (sig) {
	case SIGHUP:
		info("received %s, rereading configuration file %s",
		    signal2str(sig), conffile);
		/* conffile was initialized prior to the sighandler */
		if ((tmp_modules = readconf(CONFFILE)) == NULL) {
			error("error reading configuration file %s, keeping "
			    "old configuration", conffile);
		} else {
			new_modules = tmp_modules;
			if (shutdown_pid) {
				info("waiting for shutdown process to "
				    "terminate");
				shutdown_state = INACTIVE;
				stop_shutdown();
				(void) reap_dead_child(&shutdown_pid, true,
				    false);
			}
		}
		break;
	case SIGCHLD:
		debug("received %s", signal2str(sig));
		if (reap_dead_child(&shutdown_pid, false, conf->commit_suicide)
		    && shutdown_state != INACTIVE)
			/*
			 * In case of recovery or SIGHUP, shutdown_state was set
			 * to INACTIVE prior to killing the child.  In all other
			 * cases (that is, if either the shutdown was completed
			 * or the user terminated the process), we'll set
			 * shutdown_state to DONE.
			 */
			shutdown_state = DONE;
		break;
	case SIGUSR1:
		if (shutdown_pid) {
			info("received %s, stopping shutdown", signal2str(sig));
			stop_shutdown();
		} else {
			/* conf was initialized prior to the sighandler */
			info("received %s, starting shutdown", signal2str(sig));
			if ((shutdown_pid = start_shutdown(CONF)) != -1)
				shutdown_state = USER;
			else
				shutdown_pid = 0;
		}
		break;
	case SIGUSR2:
		if (!shutdown_pid) {
			/* conf was initialized prior to the sighandler */
			info("received %s, starting immediate shutdown",
			    signal2str(sig));
			if ((shutdown_pid = start_shutdown(CONF)) != -1)
				shutdown_state = USER;
			else
				shutdown_pid = 0;
		} else
			info("received %s", signal2str(sig));
		if (shutdown_pid && kill(shutdown_pid, SIGUSR2) == -1)
			error("cannot send SIGUSR2 signal to shutdown process "
			    "%d: %m", shutdown_pid);
		break;
	/* terminating signals */
	case SIGINT:	/* FALLTHROUGH */
	case SIGTERM:
		info("received %s, cleaning up", signal2str(sig));
		if (!termination_running) {
			termination_running = 1;
			if (shutdown_pid) {
				info("waiting for shutdown process to "
				    "terminate");
				stop_shutdown();
				(void) reap_dead_child(&shutdown_pid, true,
				    false);
			}
			if (pidfd != -1) {
				/* pidfile is defined if pidfd != -1 */
				if (lseek(pidfd, 0, SEEK_SET) == -1)
					error("lseek(2)ing pidfile failed: %m");
#if HAVE_LOCKF
				if (lockf(pidfd, F_ULOCK, 0) == -1)
					error("unlocking pidfile failed: %m");
#endif
				if (close(pidfd) == -1)
					error("closing pidfile failed: %m");
				if (unlink(PIDFILE) == -1)
					error("unlinking pidfile failed: %m");
				pidfd = -1;
			}
			notice("terminating due to signal %s", signal2str(sig));
			close_log();
			signal(sig, SIG_DFL);
			(void) kill(0, sig);	/* default action */
		}
		break;
	default:
		/* should never happen */
		error("signal %d not handled", sig);
	}
}
